-- Dokumentation siehe
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/TCimEvent
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Konfiguration_der_Benutzerrechte
--
-- Wenn Tabellen beim Prodat-Login noch nicht existieren, dann wird in TDM1.DataBase1AfterConnect "Z0 role-system.sql" geladen und ausgeführt.


/*
DROP TABLE knownactions CASCADE;
DROP TABLE userroles CASCADE;
DROP TABLE rolepermissions CASCADE;
DROP TABLE loginrole CASCADE;
DROP TABLE rule CASCADE;
DROP FUNCTION IF EXISTS loginrole__b_iud() CASCADE;
DROP FUNCTION IF EXISTS loginrole__a_iu() CASCADE;
DROP FUNCTION IF EXISTS CreateDefaultUserRoles(IN dblogin VARCHAR) CASCADE;
DROP FUNCTION IF EXISTS DeleteUserRoles(IN dblogin VARCHAR) CASCADE;
DROP FUNCTION IF EXISTS RenameUserRoles(IN olddblogin VARCHAR, IN newdblogin VARCHAR) CASCADE;
DROP FUNCTION IF EXISTS SyncGroupIntoRoles(IN inGroup VARCHAR) CASCADE;
DROP FUNCTION IF EXISTS SyncGroupsForUserIntoRoles(IN inUser VARCHAR) CASCADE;
DROP FUNCTION IF EXISTS isRoleDistinctFromGroup(IN inRole VARCHAR) CASCADE;
DROP FUNCTION IF EXISTS llv__a_iud_synchUserrights() CASCADE;
DROP FUNCTION IF EXISTS GetRulesByLogin(IN dbLogin VARCHAR, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _ebene INTEGER, OUT _ru_id INTEGER) CASCADE;
DROP FUNCTION IF EXISTS GetRulesByRoleId(IN roleid INTEGER, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _ebene INTEGER, OUT _ru_id INTEGER) CASCADE;
DROP FUNCTION IF EXISTS GetRolesByLogin(IN dblogin VARCHAR, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _lro_parent INTEGER, OUT _ebene INTEGER, OUT _lro_shared BOOLEAN) CASCADE;
DROP FUNCTION IF EXISTS GetRolesByRoleID(IN roleid INTEGER, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _lro_parent INTEGER, OUT _ebene INTEGER, OUT _lro_shared BOOLEAN) CASCADE;
*/



-- Welche Regeln erlauben oder verbieten bestimmte Sachen
CREATE TABLE rule
( ru_id         SERIAL PRIMARY KEY,
  ru_parent     INTEGER REFERENCES rule ON UPDATE CASCADE ON DELETE CASCADE,  -- Nur zum Zusammenstellen und verwalten.
  ru_bez        VARCHAR(100) NOT NULL,          -- Name halt
  ru_textnr     INTEGER REFERENCES text0 ON UPDATE CASCADE,
  ru_type       VARCHAR(50) NOT NULL,           -- Was wo geprueft wird. Im Query oder bei Aufruf von Delphi-Funktionen/Modulen etc...
  ru_object     VARCHAR(50),                    -- Auf welches Object sich das bezieht
  ru_subobject  VARCHAR(50),                    -- ...
  ru_action     VARCHAR(100),                   -- Welche Zugriffe die Regel erlaubt (oder verbietet)
  ru_condition  TEXT,                           -- Pruefbedingungen (insert_by = current_user)
  ru_isActive   BOOLEAN NOT NULL DEFAULT TRUE,  -- Ob die Regel aktiv ist und geprüft werden muss
  ru_isVisible  BOOLEAN NOT NULL DEFAULT TRUE,  -- Unsichtbare Regeln. Damit können wir z.Bsp. nicht freigeschaltete Module sperren.
  ru_forbidden  BOOLEAN NOT NULL DEFAULT FALSE, -- True = Aufgeführte actions sind verboten, alles andere erlaubt.
  UNIQUE(ru_bez, ru_type)
);



--Welche Rollen erben von welchen anderen
CREATE TABLE loginrole
( lro_id        SERIAL PRIMARY KEY,
  lro_bez       VARCHAR(50) NOT NULL UNIQUE,      -- Gruppennmen sind CaseSensitiv und Benutzernamen CaseInsensitiv, Rollen auf DB immer case sensitiv
  lro_textnr    INTEGER REFERENCES text0 ON UPDATE CASCADE,
  lro_parent    INTEGER REFERENCES loginrole ON UPDATE CASCADE ON DELETE CASCADE,
  lro_shared    BOOLEAN NOT NULL DEFAULT FALSE,   -- Wenn True, dann darf diese Rolle von mehreren Benutzern genutzt werden.
  lro_isDBgroup BOOLEAN DEFAULT FALSE,            -- Wenn Rolle aus pg_group; fürs Verhindern von Strukturänderungen durch UR2
  lro_desc      TEXT,                             -- Beschreibung der Rolle
  CHECK  (lro_bez > ''),
  CHECK  (lro_shared OR NOT (lro_shared AND lro_isDBgroup))  -- lro_isDBgroup nur bei Shared-Rollen
);

INSERT INTO loginrole(lro_id, lro_bez, lro_shared, lro_desc) VALUES (1, 'Alle Benutzer', TRUE, 'Standardrolle für alle Benutzer');
SELECT setval('loginrole_lro_id_seq', 1);

CREATE OR REPLACE FUNCTION loginrole__b_iud() RETURNS TRIGGER AS $$
BEGIN
  --Nutzerrolle "Alle Benutzer" darf nicht gelöscht werden.
  IF tg_op = 'DELETE' AND old.lro_id = 1 THEN
    RAISE EXCEPTION '% "%" %', lang_Text(12275), old.lro_bez, lang_text(12276);  --, old.lro_bez; --Die Benutzerrolle "%", darf nicht entfernt werden.
  END IF;

  IF (tg_op = 'INSERT' OR tg_op = 'UPDATE') AND new.lro_parent = 1 THEN
    RAISE EXCEPTION '%', lang_Text(17197);  --Benutzerrolle "Alle Benutzer" kann keine untergeordneten Rollen besitzen
  END IF;

  IF (tg_op = 'INSERT' OR tg_op = 'UPDATE')
      AND ((new.lro_isDBgroup AND new.lro_parent IS NOT NULL)                         -- DB-Gruppe wird irgendwo geschachtelt
        OR (SELECT lro_isDBgroup FROM loginrole WHERE lro_id = new.lro_parent)) THEN  -- DB-Gruppe wird parent
    RAISE EXCEPTION '%', lang_Text(16158);  -- Eine Datenbankgruppe kann nicht verschachtelt werden.
  END IF;

  IF TG_OP <> 'DELETE' THEN
    RETURN new;
  ELSE
    RETURN old;
  END IF;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER loginrole__b_iud
  BEFORE INSERT OR UPDATE OR DELETE
  ON loginrole
  FOR EACH ROW
  EXECUTE PROCEDURE loginrole__b_iud();



-- Synch Datenbankgruppen
CREATE OR REPLACE FUNCTION loginrole__a_iu() RETURNS TRIGGER AS $$
BEGIN
  new.lro_isDBgroup := EXISTS(SELECT TRUE FROM pg_group WHERE groname = new.lro_bez);
  UPDATE loginrole SET lro_isDBgroup = new.lro_isDBgroup WHERE LOWER(lro_bez) = LOWER(new.lro_bez) AND lro_shared;

  -- Wenn UR2-Rolle umbenannt wird und DB-Gruppe war, entferne alle Beziehungen zur Gruppe
  IF TG_OP = 'UPDATE' AND old.lro_isDBgroup THEN  -- im Trigger: nur UPDATE OF lro_bez
    DELETE FROM userroles WHERE uro_lro_id = (SELECT lro_id FROM loginrole WHERE LOWER(lro_bez) = LOWER(new.lro_bez) AND lro_isDBgroup);
  END IF;

  -- Wenn neue UR2-Gruppe DB-Gruppe ist, baue Beziehungen auf
  IF new.lro_isDBgroup THEN
    PERFORM SyncGroupIntoRoles(new.lro_bez);
  END IF;

  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER loginrole__a_iu
  AFTER INSERT OR UPDATE OF lro_bez
  ON loginrole
  FOR EACH ROW
  WHEN (new.lro_shared)
  EXECUTE PROCEDURE loginrole__a_iu();



--Für welche Rolle gelten welche Regeln
CREATE TABLE rolepermissions
( rp_id       SERIAL PRIMARY KEY,
  rp_lro_id   INTEGER NOT NULL REFERENCES loginrole  ON UPDATE CASCADE ON DELETE CASCADE,
  rp_ru_id    INTEGER NOT NULL REFERENCES rule       ON UPDATE CASCADE ON DELETE CASCADE,
  UNIQUE (rp_lro_id, rp_ru_id)
);

--CREATE UNIQUE INDEX rolepermissions_rp_lro_id_rp_ru_id ON rolepermissions(rp_lro_id, rp_ru_id);



--Welcher Nutzer hat welche Rollen. (Jeder Nutzer ist in Rolle "Alle User" und "LoginName" ... Nutzer mit Login 'lgerlach' nimmt also automatisch (als einziger) Rolle 'lgerlach' ein.
CREATE TABLE userroles
( uro_id          SERIAL PRIMARY KEY,
  uro_db_usename  VARCHAR(10) NOT NULL,  -- 10 = Größe von llv.ll_db_usename
  uro_lro_id      INTEGER NOT NULL REFERENCES loginrole ON UPDATE CASCADE ON DELETE CASCADE,
  UNIQUE (uro_db_usename, uro_lro_id)
);

--CREATE UNIQUE INDEX userroles_usename_role ON userroles(uro_db_usename, uro_lro_id);



--Welche Actions sind als rtObject für rtAction bekannt (können manuell eingetagen werden oder werden automatisch erstellt, wärend sie im Programm verwendet/geprüft werden)
CREATE TABLE knownactions
( ka_id       SERIAL PRIMARY KEY,
  ka_object   VARCHAR(50) NOT NULL UNIQUE
);



CREATE OR REPLACE FUNCTION CreateDefaultUserRoles(IN dblogin VARCHAR) RETURNS VOID AS $$
DECLARE id INTEGER;
BEGIN
  -- Wenn es noch keine Rolle mit Rolle=Benutzername gibt, anlegen.
  IF NOT EXISTS(SELECT TRUE FROM loginrole WHERE LOWER(lro_bez) = LOWER(dblogin) AND NOT lro_shared) THEN
    INSERT INTO loginrole (lro_bez, lro_shared, lro_desc) VALUES (dblogin, FALSE, 'Benutzerrolle');
  END IF;

  --Noch keine Nutzerrolle zugewiesen?
  IF NOT EXISTS(SELECT TRUE FROM userroles JOIN loginrole ON uro_lro_id = lro_id WHERE LOWER(lro_bez) = LOWER(dblogin) AND NOT lro_shared) THEN
    SELECT lro_id INTO id FROM loginrole WHERE LOWER(lro_bez) = LOWER(dblogin) AND NOT lro_shared;
    INSERT INTO userroles (uro_db_usename, uro_lro_id) VALUES (dblogin, id);
  END IF;

  -- Wenn Benutzer noch nicht in Rolle "Alle Benutzer" ist (ID 1)
  IF NOT EXISTS(SELECT TRUE FROM loginrole JOIN userroles ON uro_lro_id = lro_id WHERE lro_id = 1 AND LOWER(uro_db_usename) = LOWER(dblogin)) THEN
    INSERT INTO userroles (uro_db_usename, uro_lro_id) VALUES (dblogin, 1);
  END IF;

  RETURN;
END $$ LANGUAGE plpgsql VOLATILE;



-- Verwendung in llv__a_iud_synchUserrights()
CREATE OR REPLACE FUNCTION DeleteUserRoles(IN dblogin VARCHAR) RETURNS VOID AS $$
DECLARE id INTEGER;
BEGIN
  SELECT lro_id INTO id FROM loginrole WHERE LOWER(lro_bez) = LOWER(dblogin) AND NOT lro_shared;
  IF id IS NOT NULL THEN  -- loginrole existiert
    -- Regeln für die Rollen entfernen
    DELETE FROM rolepermissions WHERE rp_lro_id = id;

    -- Rollenzuordnung entfernen
    DELETE FROM userroles WHERE LOWER(uro_db_usename) = LOWER(dblogin);

    -- Rolle selbst entfernen
    DELETE FROM loginrole WHERE lro_id = id;
  END IF;
  RETURN;
END $$ LANGUAGE plpgsql VOLATILE;



-- Verwendung in llv__a_iud_synchUserrights()
CREATE OR REPLACE FUNCTION RenameUserRoles(IN olddblogin VARCHAR, IN newdblogin VARCHAR) RETURNS VOID AS $$
BEGIN
  IF NOT EXISTS(SELECT TRUE FROM loginrole WHERE LOWER(lro_bez) = LOWER(newdblogin) AND NOT lro_shared) THEN
    UPDATE loginrole SET lro_bez = newdblogin WHERE lro_id = (SELECT lro_id FROM loginrole WHERE LOWER(lro_bez) = LOWER(olddblogin) AND NOT lro_shared);
    UPDATE userroles SET uro_db_usename = newdblogin WHERE LOWER(uro_db_usename) = LOWER(olddblogin);
  ELSE
    RAISE EXCEPTION '%', lang_Text(16159); -- Neuer Login-Name schon vergeben.
  END IF;
  RETURN;
END $$ LANGUAGE plpgsql VOLATILE;



-- Synchronisation von Benutzern in der Datenbankgruppe zu UserRights2
-- inGroup/lro_bez ist case sensitive
-- Verschachtelungen nicht gelöst:
/* SELECT pg_group.groname, g2.groname FROM pg_group
   JOIN pg_group AS g2 ON pg_group.grosysid = ANY(g2.grolist)
   WHERE pg_group.groname = 'SYS.QAB-Admin'*/
CREATE OR REPLACE FUNCTION SyncGroupIntoRoles(IN inGroup VARCHAR) RETURNS VOID AS $$
DECLARE name RECORD;
BEGIN
  IF EXISTS(SELECT TRUE FROM pg_group WHERE groname = inGroup)  -- Quelle prüfen
      AND EXISTS(SELECT TRUE FROM loginrole WHERE lro_bez = inGroup AND lro_shared) THEN  -- Ziel prüfen

    -- alte Benutzerzuordnung weg
    DELETE FROM userroles WHERE uro_lro_id = (SELECT lro_id FROM loginrole WHERE lro_bez = inGroup AND lro_shared);
    UPDATE loginrole SET lro_isDBgroup = TRUE WHERE lro_bez = inGroup AND lro_shared;

    -- neue Benutzerzuordnung hinzu
    FOR name IN SELECT usename FROM pg_user JOIN pg_group ON usesysid = ANY(grolist) WHERE groname = inGroup ORDER BY usename
    LOOP
      -- alle Benutzer aus der DB-Gruppe in UR2
      IF NOT EXISTS(SELECT TRUE FROM loginrole WHERE LOWER(lro_bez) = LOWER(name.usename) AND NOT lro_shared) THEN
        PERFORM CreateDefaultUserRoles(CAST(name.usename AS VARCHAR));
      END IF;

      -- alle Beziehungen entsprechend DB-Gruppe aufbauen
      INSERT INTO userroles (uro_db_usename, uro_lro_id)
        SELECT CAST(name.usename AS VARCHAR), lro_id FROM loginrole WHERE lro_bez = inGroup AND lro_shared;
    END LOOP;
  END IF;
  RETURN;
END $$ LANGUAGE plpgsql VOLATILE;



-- Synchronasation der DB-Gruppen zu UR2 eines Benutzers
CREATE OR REPLACE FUNCTION SyncGroupsForUserIntoRoles(IN inUser VARCHAR) RETURNS VOID AS $$
BEGIN
  IF EXISTS(SELECT TRUE FROM pg_user WHERE LOWER(usename) = LOWER(inUser)) AND
     EXISTS(SELECT TRUE FROM loginrole WHERE LOWER(lro_bez) = LOWER(inUser) AND NOT lro_shared) THEN

    DELETE FROM userroles
      WHERE LOWER(uro_db_usename) = LOWER(inUser)
        AND uro_lro_id IN (SELECT lro_id FROM loginrole
                             JOIN pg_group ON groname = lro_bez AND lro_isDBgroup
                             JOIN pg_user ON usesysid = ANY(grolist)
                             WHERE LOWER(usename) = LOWER(inUser));

    INSERT INTO userroles (uro_db_usename, uro_lro_id)
      SELECT inUser, lro_id
        FROM loginrole
        JOIN pg_group ON lro_bez = groname AND lro_isDBgroup
        JOIN pg_user ON usesysid = ANY(grolist)
        WHERE LOWER(usename) = LOWER(inUser);
  END IF;
  RETURN;
END $$ LANGUAGE plpgsql VOLATILE;



-- Vergleiche Datenbankgruppe und entsprechende UR2-Rolle
CREATE OR REPLACE FUNCTION isRoleDistinctFromGroup(IN inRole VARCHAR) RETURNS BOOLEAN AS $$
BEGIN
  IF (SELECT lro_isDBgroup FROM loginrole WHERE lro_bez = inRole AND lro_shared) THEN
    RETURN BOOL_OR(userroles.uro_db_usename IS NULL OR pg_user.usename IS NULL)
      FROM
      (SELECT uro_db_usename
         FROM userroles
         JOIN loginrole ON lro_id = uro_lro_id
         WHERE lro_bez = inRole
      ) AS userroles
      FULL JOIN
      (SELECT usename
         FROM pg_user
         JOIN pg_group ON usesysid = ANY(grolist)
         WHERE groname = inRole
      ) AS pg_user
      ON pg_user.usename = userroles.uro_db_usename;
  ELSE
    RETURN FALSE;
  END IF;
END $$ LANGUAGE plpgsql STABLE;



-- llv-Trigger für Mitarbeiterverzeichnis synch UR2
CREATE OR REPLACE FUNCTION llv__a_iud_synchUserrights() RETURNS TRIGGER AS $$
BEGIN
  IF TG_OP = 'INSERT' THEN
    IF new.ll_db_login THEN
      PERFORM CreateDefaultUserRoles(new.ll_db_usename);
      PERFORM SyncGroupsForUserIntoRoles(new.ll_db_usename);
    END IF;
    RETURN new;
  END IF;

  IF TG_OP = 'UPDATE' THEN
    -- Neu wenn Loginflag gesetzt
    IF new.ll_db_login AND NOT old.ll_db_login THEN
      PERFORM CreateDefaultUserRoles(new.ll_db_usename);
      PERFORM SyncGroupsForUserIntoRoles(new.ll_db_usename);
    END IF;

    -- Löschen wenn Loginflag auf falsch für genau diesen Nutzer gesetzt
    IF NOT new.ll_db_login AND old.ll_db_login AND new.ll_db_usename = old.ll_db_usename THEN
      PERFORM DeleteUserRoles(new.ll_db_usename);
    END IF;

    -- Umbenennen wenn Loginflag schon war und Name sich geändert
    IF new.ll_db_login AND old.ll_db_login AND new.ll_db_usename <> old.ll_db_usename THEN
      PERFORM RenameUserRoles(old.ll_db_usename, new.ll_db_usename);
    END IF;
    RETURN new;
  END IF;

  IF TG_OP = 'DELETE' THEN
    IF old.ll_db_login THEN
      PERFORM DeleteUserRoles(old.ll_db_usename);
    END IF;
    RETURN old;
  END IF;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER llv__a_iud_synchUserrights
  AFTER INSERT OR UPDATE OR DELETE
  ON llv
  FOR EACH ROW
  EXECUTE PROCEDURE llv__a_iud_synchUserrights();



--Alle Rollen eines Nutzers
CREATE OR REPLACE FUNCTION GetRulesByLogin(IN dbLogin VARCHAR, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _ebene INTEGER, OUT _ru_id INTEGER) RETURNS SETOF RECORD AS $$
DECLARE rolrec RECORD;
        rulrec RECORD;
BEGIN
  IF current_user = 'root' THEN
    RAISE NOTICE 'GetRulesByLogin: => Suche Regeln für Login % und geerbte Rollen. ', dblogin;
  END IF;

  FOR rolrec IN
  (
    --Rekursiv durch Vererbungsbaum der Rollen laufen
    WITH RECURSIVE roles(lro_id, lro_bez, lro_parent, ebene, lro_shared) AS
    (
      --Startbedingung. "erste ebene"
      SELECT DISTINCT lro_id, lro_bez, lro_parent, 1 AS ebene, lro_shared
        FROM loginrole LEFT JOIN userroles ON uro_lro_id = lro_id
        WHERE LOWER(uro_db_usename) = LOWER(dbLogin)
      UNION
      --Rekursiv hinzugefügte Sätze
      SELECT DISTINCT l.lro_id, l.lro_bez, l.lro_parent, par.ebene+1, l.lro_shared
        FROM loginrole l INNER JOIN roles par ON par.lro_id = l.lro_parent
        WHERE ebene<32 --32 Vererbungsebenen, Anzahl der Rekursionen ist also hart beschränkt.
    )
      SELECT DISTINCT lro_id, lro_bez, lro_parent, ebene, lro_shared
        FROM roles
        ORDER BY lro_id LIMIT 1000
  )
  LOOP
    _lro_id  := rolrec.lro_id;
    _lro_bez := rolrec.lro_bez;
    _ebene   := rolrec.ebene;

    --Return ist also Rollen-ID + Bezeichnung + Welche Ebene der Rollenhierarchie + Rule-ID
    IF current_user='root' THEN  RAISE NOTICE 'GetRulesByLogin: => Rolle gefunden: %, % - Ebene: % ', rolRec.lro_id, rolRec.lro_bez, rolRec.ebene;  END IF;

    --Rekursiv durch Baum der Regeln laufen (Für jede Rolle)
    FOR rulrec IN
    (
      WITH RECURSIVE rules(ru_id, ru_bez, ru_parent, ruebene) AS
      (
        --Startbedingung. "erste ebene"
        SELECT DISTINCT ru_id, ru_bez, ru_parent, 1 as ruebene  --Regeln zur Rolle finden.
          FROM rule JOIN rolepermissions ON rp_ru_id = ru_id
          WHERE rp_lro_id = _lro_id                             --Fuer gerade Aktive Rolle.
        UNION
        --Rekursiv hinzugefügte Sätze
        SELECT DISTINCT l.ru_id, l.ru_bez, l.ru_parent, par.ruebene+1
          FROM rule l INNER JOIN rules par ON par.ru_id = l.ru_parent
          WHERE ruebene<32 --32 Vererbungsebenen, Anzahl der Rekursionen ist also hart beschränkt.
      )
        SELECT DISTINCT ru_id, ru_bez, ru_parent, ruebene
          FROM rules
          ORDER BY ru_id LIMIT 1000
    )
    LOOP
      --Return ist also Rollen-ID + Bezeichnung + Welche Ebene der Rollenhierarchie + Rule-ID
      IF current_user = 'root' THEN
        RAISE NOTICE 'GetRulesByLogin: => Regel gefunden für Rolle: %, % - Regel: %, Ebene: % - % ', _lro_id, _lro_bez, rulrec.ru_id, rulrec.ruebene, rulrec.ru_bez;
      END IF;
      _ru_id := rulrec.ru_id;
      _ebene := rulrec.ruebene;
      RETURN NEXT;
    END LOOP;
  END LOOP;

  RETURN;
END $$ LANGUAGE plpgsql STABLE;



--Alle Rollen eines Nutzers
CREATE OR REPLACE FUNCTION GetRulesByRoleId(IN roleid INTEGER, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _ebene INTEGER, OUT _ru_id INTEGER) RETURNS SETOF RECORD AS $$
DECLARE rolrec RECORD;
        rulrec RECORD;
BEGIN
  IF current_user = 'root' THEN
    RAISE NOTICE 'GetRulesByRoleId: => Suche Regeln für Rolle % und geerbte Rollen. ', roleid;
  END IF;
  FOR rolrec IN
  (
    --Rekursiv durch Vererbungsbaum der Rollen laufen
    WITH RECURSIVE roles(lro_id, lro_bez, lro_parent, ebene, lro_shared) AS
    (
      --Startbedingung. "erste ebene"
      SELECT DISTINCT lro_id, lro_bez, lro_parent, 1 AS ebene, lro_shared
        FROM loginrole LEFT JOIN userroles ON uro_lro_id = lro_id
        WHERE lro_id = roleid
      UNION
      --Rekursiv hinzugefügte Sätze
      SELECT DISTINCT l.lro_id, l.lro_bez, l.lro_parent, par.ebene+1, l.lro_shared
        FROM loginrole l INNER JOIN roles par ON par.lro_id = l.lro_parent
        WHERE ebene<32  --32 Vererbungsebenen, Anzahl der Rekursionen ist also hart beschränkt.
    )
    SELECT DISTINCT lro_id, lro_bez, lro_parent, ebene, lro_shared
      FROM roles
      ORDER BY lro_id LIMIT 1000
  )
  LOOP
    _lro_id  := rolrec.lro_id;
    _lro_bez := rolrec.lro_bez;
    _ebene   := rolrec.ebene;

    --Return ist also Rollen-ID + Bezeichnung + Welche Ebene der Rollenhierarchie + Rule-ID
    IF current_user='root' THEN
      RAISE NOTICE 'GetRulesByRoleId: => Rolle gefunden: %, % - Ebene: % ', rolRec.lro_id, rolRec.lro_bez, rolRec.ebene;
    END IF;

    --Rekursiv durch Baum der Regeln laufen (Für jede Rolle)
    FOR rulrec IN
    (
      WITH RECURSIVE rules(ru_id, ru_bez, ru_parent, ruebene) AS
      (
        --Startbedingung. "erste ebene"
        SELECT DISTINCT ru_id, ru_bez, ru_parent, 1 as ruebene  --Regeln zur Rolle finden.
          FROM rule JOIN rolepermissions ON rp_ru_id = ru_id
          WHERE rp_lro_id = _lro_id                                             --Fuer gerade Aktive Rolle.
        UNION
        --Rekursiv hinzugefügte Sätze
        SELECT DISTINCT l.ru_id, l.ru_bez, l.ru_parent, par.ruebene+1
          FROM rule l INNER JOIN rules par ON par.ru_id = l.ru_parent
          WHERE ruebene<32  --32 Vererbungsebenen, Anzahl der Rekursionen ist also hart beschränkt.
      )
      SELECT DISTINCT ru_id, ru_bez, ru_parent, ruebene
        FROM rules
        ORDER BY ru_id LIMIT 1000
    )
    LOOP
      --Return ist also Rollen-ID + Bezeichnung + Welche Ebene der Rollenhierarchie + Rule-ID
      IF current_user = 'root' THEN
        RAISE NOTICE 'GetRulesByRoleId: => Regel gefunden für Rolle: %, % - Regel: %, Ebene: % - % ', _lro_id, _lro_bez, rulrec.ru_id, rulrec.ruebene, rulrec.ru_bez;
      END IF;

      _ru_id := rulrec.ru_id;
      RETURN NEXT;
    END LOOP;
  END LOOP;

  RETURN;
END $$ LANGUAGE plpgsql STABLE;



--Alle Rollen eines Nutzers
CREATE OR REPLACE FUNCTION GetRolesByLogin(IN dblogin VARCHAR, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _lro_parent INTEGER,
                                           OUT _ebene INTEGER, OUT _lro_shared BOOLEAN) RETURNS SETOF RECORD AS $$
DECLARE r RECORD;
BEGIN
  FOR r IN
  (
    --Rekursiv durch Vererbungsbaum laufen
    WITH RECURSIVE rolesOfUser(lro_id, lro_bez, lro_parent, ebene, lro_shared) AS
    (
      --Startbedingung. "erste ebene"
      SELECT lro_id, lro_bez, lro_parent, 1 AS ebene, lro_shared
        FROM loginrole LEFT JOIN userroles ON uro_lro_id = lro_id
        WHERE LOWER(uro_db_usename) = LOWER(dblogin)
      UNION
      --Rekursiv hinzugefügte Sätze
      SELECT l.lro_id, l.lro_bez, l.lro_parent, par.ebene+1, l.lro_shared
        FROM loginrole l INNER JOIN rolesOfUser par ON par.lro_id = l.lro_parent
        WHERE ebene<32  --32 Vererbungsebenen, Anzahl der Rekursionen ist also hart beschränkt.
    )

    SELECT DISTINCT lro_id, lro_bez, lro_parent, ebene, lro_shared
      FROM rolesOfUser
      ORDER BY lro_id LIMIT 1000
  )
  LOOP
    _lro_id     := r.lro_id;
    _lro_bez    := r.lro_bez;
    _lro_parent := r.lro_parent;
    _ebene      := r.ebene;
    _lro_shared := r.lro_shared;
    RETURN NEXT;
  END LOOP;
  RETURN;
END $$ LANGUAGE plpgsql STABLE;



--Alle Unterrollen einer bestimmten Rolle.
CREATE OR REPLACE FUNCTION GetRolesByRoleID(IN roleid INTEGER, OUT _lro_id INTEGER, OUT _lro_bez VARCHAR, OUT _lro_parent INTEGER,
                                            OUT _ebene INTEGER, OUT _lro_shared BOOLEAN) RETURNS SETOF RECORD AS $$
DECLARE r RECORD;
BEGIN
  FOR r IN
  (
    --Rekursiv durch Vererbungsbaum laufen
    WITH RECURSIVE roles(lro_id, lro_bez, lro_parent, ebene, lro_shared) AS
    (
      --Startbedingung. "erste ebene"
      SELECT DISTINCT lro_id, lro_bez, lro_parent, 1 AS ebene, lro_shared
        FROM loginrole LEFT JOIN userroles ON uro_lro_id = lro_id
        WHERE lro_id = roleid
      UNION
      --Rekursiv hinzugefügte Sätze
      SELECT DISTINCT l.lro_id, l.lro_bez, l.lro_parent, par.ebene+1, l.lro_shared
        FROM loginrole l INNER JOIN roles par ON par.lro_id = l.lro_parent
        WHERE ebene<32  --32 Vererbungsebenen, Anzahl der Rekursionen ist also hart beschränkt.
    )

    SELECT DISTINCT lro_id, lro_bez, lro_parent, ebene, lro_shared
      FROM roles
      ORDER BY lro_id LIMIT 1000
  )
  LOOP
    _lro_id     := r.lro_id;
    _lro_bez    := r.lro_bez;
    _lro_parent := r.lro_parent;
    _lro_shared := r.lro_shared;
    _ebene      := r.ebene;

    RETURN NEXT;
  END LOOP;
  RETURN;
END $$ LANGUAGE plpgsql STABLE;

-- #21159 BEGIN
-- view which shows recursive for roles the membership of that role
DROP VIEW IF EXISTS TSystem.rolemembership_view;
CREATE OR REPLACE VIEW TSystem.rolemembership_view AS

  WITH RECURSIVE membership_tree(grpid, userid, parentgrp, depth, path) AS
  (
    -- Get all roles and list them as their own group as well
    SELECT
      pg_roles.oid, pg_roles.oid, pg_roles.oid, 1 AS DEPTH, ''::name AS path
    FROM
      pg_roles
    UNION ALL
    -- Now add all group memberships
    SELECT
      m_1.roleid, t_1.userid, t_1.grpid, depth+1 AS DEPTH, IfThen( (path = ''), '', (path || '>') ) || pg_get_userbyid(t_1.grpid) AS path
    FROM
      pg_auth_members m_1, membership_tree t_1
    WHERE
      m_1.member = t_1.grpid
      AND depth <= 10         -- Begrenzung der Rekursionstiefe, um Endlosschleifen zu vermmeiden
  )
  SELECT DISTINCT
    t.userid as role_oid, r.rolname AS role_name, t.grpid as membergroup_oid, m.rolname AS membergroup_name, pg_get_userbyid(parentgrp) AS direct_groupparent_name, DEPTH, IfThen( (path = ''), '', (path || '>') ) || m.rolname AS path
  FROM
    membership_tree t, pg_roles r, pg_roles m
  WHERE
    t.grpid = m.oid AND t.userid = r.oid
  ORDER BY
    r.rolname, depth, path
;

-- uses TSystem.rolemembership_view, to calculate access-right for a role, based on a list of allowed and denied roles
CREATE OR REPLACE FUNCTION TSystem.role_allowed(rolename VARCHAR, rolenamelist_allow VARCHAR, rolenamelist_deny VARCHAR) RETURNS BOOLEAN AS $$
DECLARE
  _IsAllowed BOOLEAN;
  _IsDenied BOOLEAN;

BEGIN
  SELECT
    COUNT(*) > 0
  FROM
    TSystem.rolemembership_view
  WHERE
    role_name = rolename
    and membergroup_name = ANY(STRING_TO_ARRAY(rolenamelist_allow, ',', ''))
  INTO
    _IsAllowed;

  SELECT
    COUNT(*) > 0
  FROM
    TSystem.rolemembership_view
  WHERE
    role_name = rolename
    and membergroup_name = ANY(STRING_TO_ARRAY(rolenamelist_deny, ',', ''))
  INTO
    _IsDenied;

  RETURN (_IsAllowed and not(_IsDenied));
END$$ LANGUAGE plpgsql VOLATILE;
-- #21159 END

-- keine leeren Statements am Ende vom Erstellen der DB erlaubt.
SELECT TRUE;
